/*-----------------------------------------------------------------------------
 * EcLink.h                 
 * Copyright                acontis technologies GmbH, Weingarten, Germany
 * Response                 Stefan Zintgraf
 * Description              EtherCAT Master link layer interface
 *---------------------------------------------------------------------------*/

#ifndef INC_ECLINK
#define INC_ECLINK

/*-INCLUDES------------------------------------------------------------------*/

#ifndef INC_ECTYPE
#include <EcType.h>
#endif
#ifndef INC_ECOS
#include <EcOs.h>
#endif
#ifndef INC_ECINTERFACECOMMON
#include <EcInterfaceCommon.h>
#endif

#ifndef ATEMLL_API
#define ATEMLL_API
#endif

#ifdef __cplusplus
extern "C" {
#endif


/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#define EC_PACKED __attribute__((aligned(4), packed))

#if (defined __RCX__)
#pragma pack (4)
#endif

#elif defined _DIAB_TOOL

#define EC_PACKED __attribute__((aligned(4), packed))

#elif defined _MSC_VER  /* MICROSOFT */

#define EC_PACKED
#pragma pack (push, 4)
#pragma warning(disable:4514)

#elif defined __MET__

#define EC_PACKED
/*#pragma pack (push, 4)*/
#pragma Push_align_members(4)

#else
#error Please check pack settings for this compiler!!!
#endif


/*-DEFINES-------------------------------------------------------------------*/

#define LINK_LAYER_DRV_DESC_PATTERN         0x11aaddaa
#define LINK_LAYER_DRV_DESC_VERSION         0x00020200      /* version 2.2 */
                                            
#define LINK_LAYER_DEV_PARAM_PATTERN        0x11aaaaaa
#define LINK_LAYER_DEV_PARAM_VERSION        0x00020000      /* version 2.0 */
                                            
#define MAX_DRIVER_IDENT_LEN                40              /* maximum length for a driver ident string */
                                            
#define EC_LINKNOTIFY_GENERIC               0x00000000
	                                  
#define EC_LINKNOTIFY_LINKSTATECHGD         (EC_LINKNOTIFY_GENERIC | 0x1)
#define EC_LINKNOTIFY_TXMEMORYCHGD          (EC_LINKNOTIFY_GENERIC | 0x2)
                                            
#define EC_LINKIOCTL_GENERIC                0x00000000

#define EC_LINKIOCTL_INTERRUPTENABLE        (EC_LINKIOCTL_GENERIC  | 0x1)
#define EC_LINKIOCTL_PROMISCOUSMODE         (EC_LINKIOCTL_GENERIC  | 0x2)
#define EC_LINKIOCTL_SETMULTICASTADDR       (EC_LINKIOCTL_GENERIC  | 0x3)
#define EC_LINKIOCTL_CLRMULTICASTADDR       (EC_LINKIOCTL_GENERIC  | 0x4)
#define EC_LINKIOCTL_SETLINKMODE            (EC_LINKIOCTL_GENERIC  | 0x5)
#define EC_LINKIOCTL_GETLINKMODE            (EC_LINKIOCTL_GENERIC  | 0x6)
#define EC_LINKIOCTL_RESTART                (EC_LINKIOCTL_GENERIC  | 0x7)
#define EC_LINKIOCTL_SET_LINKMODE           (EC_LINKIOCTL_GENERIC  | 0x8)   /* set mode to polling or interrupt */
#define EC_LINKIOCTL_SET_ALLOC_SENDFRAME    (EC_LINKIOCTL_GENERIC  | 0x9)   /* for debug purposes: enable/disable EcLinkAllocSendFrame() support */
#define EC_LINKIOCTL_FORCELINKMODE          (EC_LINKIOCTL_GENERIC  | 0xA)   /* Bit 0=Autoneg.;1=FullDuplex;16-31=LinkSpeed 1=10,2=100,3=1000MBit */
#define EC_LINKIOCTL_GETINFOLIST            (EC_LINKIOCTL_GENERIC  | 0xB)   /* query EC_T_LINK_INFOLIST */

/* The following IO-Controls are mandatory for LinkLayer in interrupt mode */
#define EC_LINKIOCTL_REGISTER_FRAME_CLBK    (EC_LINKIOCTL_GENERIC  | 0x10)  /* registers the receive frame callback */
#define EC_LINKIOCTL_UNREGISTER_FRAME_CLBK  (EC_LINKIOCTL_GENERIC  | 0x11)  /* unregisters the receive frame callback */ 

/*-TYPEDEFS------------------------------------------------------------------*/

typedef enum _EC_T_LINKSTATUS 
{        
    eLinkStatus_UNDEFINED = 0,
    eLinkStatus_OK,
    eLinkStatus_DISCONNECTED,
    eLinkStatus_HALFDUPLEX,

    /* Borland C++ datatype alignment correction */
    eLinkStatus_BCppDummy   = 0xFFFFFFFF
} EC_T_LINKSTATUS;

typedef enum _EC_T_ALLOCSENDFRAME_MODE
{        
    EcLinkAllocSendFrameMode_UNDEFINED = 0,
    EcLinkAllocSendFrameMode_ENABLED,
    EcLinkAllocSendFrameMode_DISABLED,
    EcLinkAllocSendFrameMode_RANDOM,

    /* Borland C++ datatype alignment correction */
    EcLinkAllocSendFrameMode_BCppDummy   = 0xFFFFFFFF
} EC_T_ALLOCSENDFRAMEMODE;

struct _EC_T_LINK_NOTIFYPARMS;
struct _EC_T_LINK_FRAMEDESC;

typedef EC_T_DWORD (*EC_T_RECEIVEFRAMECALLBACK)(EC_T_VOID* pvContext, struct _EC_T_LINK_FRAMEDESC* pLinkFrameDesc, EC_T_BOOL* pbFrameProcessed);
typedef EC_T_DWORD (*EC_T_LINK_NOTIFY)(EC_T_DWORD dwCode, struct _EC_T_LINK_NOTIFYPARMS* pParms);

/* EtherCAT notify parameters */

typedef struct _EC_T_LINK_NOTIFYPARMS
{
    EC_T_VOID*      pvContext;			/*< in  context */
    EC_T_BYTE*      pbyInBuf;            /*< in  input data buffer */
    EC_T_DWORD      dwInBufSize;        /*< in  size of input data buffer in byte */
    EC_T_BYTE*      pbyOutBuf;          /*< out output data buffer */
    EC_T_DWORD      dwOutBufSize;       /*< in  size of output data buffer in byte */
    EC_T_DWORD*     pdwNumOutData;      /*< out number of output data bytes stored in output data buffer */
} EC_PACKED EC_T_LINK_NOTIFYPARMS;

/* EtherCAT link frame descriptor */

typedef struct _EC_T_LINK_FRAMEDESC
{
    EC_T_VOID*          pvContext;          /*< link layer context */
    EC_T_BYTE*          pbyFrame;           /*< frame data buffer */
    EC_T_DWORD          dwSize;             /*< size of the frame buffer */
    EC_T_BOOL           bBuffersFollow;     /*< if EC_TRUE try to queue next frame in link layer,
                                                if EC_FALSE fill up DMA descriptors to force immediate send */

    EC_T_DWORD*         pdwTimeStampLo;     /*< data store to store timestamp result to */
    EC_T_DWORD*         pdwTimeStampPostLo; /*< data store to store timestamp result to */
    EC_T_PVOID          pvTimeStampCtxt;    /*< context for pfnTimeStamp */
    EC_PF_TIMESTAMP     pfnTimeStamp;       /*< function if not EC_NULL called to do timestamping */
    EC_T_DWORD*         pdwLastTSResult;    /*< result code store of last time stamp call */
} EC_PACKED EC_T_LINK_FRAMEDESC;


/* Structure to register the Link Layer receive callback */
typedef struct _EC_T_LINK_FRM_RECV_CLBK
{
    EC_T_RECEIVEFRAMECALLBACK   pfFrameReceiveCallback;
    EC_T_VOID*                  pvDevice;
} EC_PACKED EC_T_LINK_FRM_RECV_CLBK;


typedef struct _EC_T_LINK_IOCTLPARMS
{
    EC_T_PBYTE      pbyInBuf;           /*< in  input data buffer */
    EC_T_DWORD      dwInBufSize;        /*< in  size of input data buffer in byte */
    EC_T_PBYTE      pbyOutBuf;          /*< out output data buffer */
    EC_T_DWORD      dwOutBufSize;       /*< in  size of output data buffer in byte */
    EC_T_DWORD*     pdwNumOutData;      /*< out number of output data bytes stored in output data buffer */ 
} EC_PACKED EC_T_LINK_IOCTLPARMS;

typedef EC_T_DWORD (*PF_EcLinkOpen)(EC_T_VOID* pvLinkParms, EC_T_RECEIVEFRAMECALLBACK pfReceiveFrameCallback, EC_T_LINK_NOTIFY pfLinkNotifyCallback, EC_T_VOID* pvContext, EC_T_VOID** ppvInstance );
typedef EC_T_DWORD (*PF_EcLinkClose)(EC_T_VOID* pvInstance);
typedef EC_T_DWORD (*PF_EcLinkSendFrame)(EC_T_VOID* pvInstance, EC_T_LINK_FRAMEDESC* pLinkFrameDesc);
typedef EC_T_DWORD (*PF_EcLinkSendAndFreeFrame)(EC_T_VOID* pvInstance, EC_T_LINK_FRAMEDESC* pLinkFrameDesc);
typedef EC_T_DWORD (*PF_EcLinkRecvFrame)(EC_T_VOID* pvInstance, EC_T_LINK_FRAMEDESC* pLinkFrameDesc);
typedef EC_T_DWORD (*PF_EcLinkAllocSendFrame)(EC_T_VOID* pvInstance, EC_T_LINK_FRAMEDESC* pLinkFrameDesc, EC_T_DWORD dwSize);
typedef EC_T_VOID  (*PF_EcLinkFreeSendFrame )(EC_T_VOID* pvInstance, EC_T_LINK_FRAMEDESC* pLinkFrameDesc);
typedef EC_T_VOID  (*PF_EcLinkFreeRecvFrame )(EC_T_VOID* pvInstance, EC_T_LINK_FRAMEDESC* pLinkFrameDesc);
typedef EC_T_DWORD      (*PF_EcLinkGetEthernetAddress)(EC_T_VOID* pvInstance, EC_T_BYTE* pbyEthernetAddress);
typedef EC_T_LINKSTATUS (*PF_EcLinkGetStatus)(EC_T_VOID* pvInstance);
typedef EC_T_DWORD      (*PF_EcLinkGetSpeed)(EC_T_VOID* pvInstance);
typedef EC_T_LINKMODE   (*PF_EcLinkGetMode)(EC_T_VOID* pvInstance);
typedef EC_T_DWORD      (*PF_EcLinkIoctl)(EC_T_VOID* pvInstance, EC_T_DWORD dwCode, EC_T_LINK_IOCTLPARMS* pParms );

typedef struct _EC_T_LINK_DEV_PARAM
{
    EC_T_DWORD                  dwValidationPattern;
    EC_T_DWORD                  dwDevParamVersion;
    EC_T_CHAR                   szDriverIdent[MAX_DRIVER_IDENT_LEN];    /* driver identification string (zero terminated) */
    EC_T_BOOL                   bForcePollingMode;                      /* obsolete */
    EC_T_BOOL                   bForceAllocSendFrame;                   /* obsolete */
    EC_T_PVOID                  pvDrvSpecificParam;                     /* pointer to additional driver specific parameters */
} EC_PACKED EC_T_LINK_DEV_PARAM;


typedef struct _EC_T_LINK_DRV_DESC_V_2_1
{
    EC_T_DWORD                  dwValidationPattern;
    EC_T_DWORD                  dwInterfaceVersion;
    EC_PF_OSDBGMSGHK            pfOsDbgMsgHook;
    PF_EcLinkOpen               pfEcLinkOpen;
    PF_EcLinkClose              pfEcLinkClose;

    /* Standard send functions */
    PF_EcLinkSendFrame          pfEcLinkSendFrame;
    PF_EcLinkSendAndFreeFrame   pfEcLinkSendAndFreeFrame;

    /* Timestamping time optimized send functions */
    PF_EcLinkSendFrame          pfEcLinkSendFrameHrt;
    PF_EcLinkSendAndFreeFrame   pfEcLinkSendAndFreeFrameHrt;

    PF_EcLinkRecvFrame          pfEcLinkRecvFrame;
    PF_EcLinkAllocSendFrame     pfEcLinkAllocSendFrame;
    PF_EcLinkFreeSendFrame      pfEcLinkFreeSendFrame ;
    PF_EcLinkFreeRecvFrame      pfEcLinkFreeRecvFrame ;
    PF_EcLinkGetEthernetAddress pfEcLinkGetEthernetAddress;
    PF_EcLinkGetStatus          pfEcLinkGetStatus;
    PF_EcLinkGetSpeed           pfEcLinkGetSpeed;
    PF_EcLinkGetMode            pfEcLinkGetMode;
    PF_EcLinkIoctl              pfEcLinkIoctl;
} EC_PACKED EC_T_LINK_DRV_DESC_V_2_1;



typedef struct _EC_T_LINK_DRV_DESC
{
    EC_T_DWORD                  dwValidationPattern;
    EC_T_DWORD                  dwInterfaceVersion;
    EC_PF_OSDBGMSGHK            pfOsDbgMsgHook;
    PF_EcLinkOpen               pfEcLinkOpen;
    PF_EcLinkClose              pfEcLinkClose;

    /* Standard send functions */
    PF_EcLinkSendFrame          pfEcLinkSendFrame;
    PF_EcLinkSendAndFreeFrame   pfEcLinkSendAndFreeFrame;
    
    /* Timestamping time optimized send functions */
    PF_EcLinkSendFrame          pfEcLinkSendFrameHrt;
    PF_EcLinkSendAndFreeFrame   pfEcLinkSendAndFreeFrameHrt;
    
    PF_EcLinkRecvFrame          pfEcLinkRecvFrame;
    PF_EcLinkAllocSendFrame     pfEcLinkAllocSendFrame;
    PF_EcLinkFreeSendFrame      pfEcLinkFreeSendFrame ;
    PF_EcLinkFreeRecvFrame      pfEcLinkFreeRecvFrame ;
    PF_EcLinkGetEthernetAddress pfEcLinkGetEthernetAddress;
    PF_EcLinkGetStatus          pfEcLinkGetStatus;
    PF_EcLinkGetSpeed           pfEcLinkGetSpeed;
    PF_EcLinkGetMode            pfEcLinkGetMode;
    PF_EcLinkIoctl              pfEcLinkIoctl;
    
    EC_T_VOID*                  pvLinkInstance;
} EC_PACKED EC_T_LINK_DRV_DESC;

typedef EC_T_DWORD (*PF_emllRegister)(EC_T_LINK_DRV_DESC* pLinkDrvDesc, EC_T_DWORD dwLinkDrvDescSize);

typedef EC_T_VOID (*PF_emllISRCallback)(EC_T_VOID* pvInstance, EC_T_VOID* pvContext);

typedef EC_T_DWORD (*PF_DOINT_HDL)(EC_T_PVOID pvLinkParms);


#define EC_LINK_INFO_DESCRIPTION_SIZE_MAX   63
typedef enum _EC_T_LINK_INFO_DATATYPE
{        
    eLinkInfoDataType_bool = 0,
    eLinkInfoDataType_byte,
    eLinkInfoDataType_word,
    eLinkInfoDataType_dword,
    eLinkInfoDataType_qword,

    /* Borland C++ datatype alignment correction */
    eLinkInfoDataType_BCppDummy   = 0xFFFFFFFF
} EC_T_LINK_INFO_DATATYPE;

typedef struct _EC_T_LINK_INFO
{
    EC_T_CHAR               szDescription[EC_LINK_INFO_DESCRIPTION_SIZE_MAX];  /*< Description text of the value */
    EC_T_LINK_INFO_DATATYPE eDataType;          /*< Datatype */
    union _EC_T_LINK_INFO_DATA
    {
        EC_T_BOOL           bData;              /*< Data as boolean */
        EC_T_BYTE           byData;             /*< Data as byte */
        EC_T_WORD           wData;              /*< Data as word */
        EC_T_DWORD          dwData;             /*< Data as dword */
        EC_T_UINT64         qwData;             /*< Data as qword */
    } EC_PACKED u;
} EC_PACKED EC_T_LINK_INFO;

typedef struct _EC_T_LINK_INFOLIST
{
    EC_T_DWORD          dwInfoCntMax;       /*< Total number of info elements available */
    EC_T_DWORD          dwInfoCntUsed;      /*< Number of info elements used */
    EC_T_LINK_INFO      aInfo[1];
} EC_PACKED EC_T_LINK_INFOLIST;


/*****************************************************************************/
/* Hardware specific link layer parameters                                   */
/*****************************************************************************/


typedef struct _EC_T_LINK_OPENPARMS_WINPCAP
{
    EC_T_BYTE                   arrIpAddress[4];
    EC_T_LINKMODE               eLinkMode;
} EC_PACKED EC_T_LINK_OPENPARMS_WINPCAP;



#define MAX_LEN_SNARF_ADAPTER_NAME  ((EC_T_DWORD)64)

typedef struct _EC_T_LINK_OPENPARMS_SNARF
{
    EC_T_WCHAR  aszAdapterName[MAX_LEN_SNARF_ADAPTER_NAME];
} EC_PACKED EC_T_LINK_OPENPARMS_SNARF;

#define MAX_LEN_ETHERLIB_ADAPTER_NAME  ((EC_T_DWORD)64)

typedef struct _EC_T_LINK_OPENPARMS_ETHERLIB
{
    EC_T_WCHAR  aszAdapterName[MAX_LEN_ETHERLIB_ADAPTER_NAME];
} EC_PACKED EC_T_LINK_OPENPARMS_ETHERLIB;


/* Intel Pro 100 family */
/* ==================== */

#define EC_LINK_TYPE_I8255x     0xA1ECA108

/* Link parameters */
typedef struct _EC_T_LINK_OPENPARMS_I8255x
{
    EC_T_DWORD                  dwType;
    EC_T_DWORD                  dwUnit;
    EC_T_DWORD                  dwIstPriority;
    EC_T_LINKMODE               eLinkMode;
    PF_DOINT_HDL                pfDoIntHandling;    /* [out]    interrupt processing routine for ext IntHandling */
    
    PF_emllISRCallback          pfIsrCallback;		/* [in]		optional called by ISR */
    EC_T_PVOID		            pvIsrCBContext;		/* [in]		ISR callback context */
} EC_PACKED EC_T_LINK_OPENPARMS_I8255x;


/* Intel Pro 1000 family */
/* ==================== */

#define EC_LINK_TYPE_I8254x     0xA1ECA10B

/* Link parameters */
typedef struct _EC_T_LINK_OPENPARMS_I8254x
{
    EC_T_DWORD                  dwType;         /* 0xA1ECA10B */
    EC_T_DWORD                  dwUnit;
    
    EC_T_DWORD                  dwIstPriority;
    EC_T_LINKMODE               eLinkMode;
    PF_DOINT_HDL                pfDoIntHandling;        /* [out]    interrupt processing routine for ext IntHandling */
    
    EC_T_DWORD                  dwRxBuffers;            /* [in]     amount of Receive  Buffers */
    EC_T_DWORD                  dwTxBuffers;            /* [in]     amount of Transmit Buffers (Frames) */
    
    PF_emllISRCallback          pfIsrCallback;          /* [in]     optional called by ISR */
    EC_T_PVOID                  pvIsrCBContext;         /* [in]     ISR callback context */
    EC_T_BOOL                   bDisableLocks;          /* [in]     Locks in LL Disabled */
} EC_PACKED EC_T_LINK_OPENPARMS_I8254x;


/* Realtek RTL8139      */
/* ==================== */

#define EC_LINK_TYPE_RTL8139     0xA1ECA109

/* Link parameters */
typedef struct _EC_T_LINK_OPENPARMS_RTL8139
{
    EC_T_DWORD                  dwType;         /* 0xA1ECA109 */
    EC_T_DWORD                  dwUnit;
                                
    EC_T_DWORD                  dwIstPriority;
    EC_T_LINKMODE               eLinkMode;
    PF_DOINT_HDL                pfDoIntHandling;        /* [out]    interrupt processing routine for ext IntHandling */

    EC_T_DWORD                  dwRxBuffers;            /* [in]     amount of Receive  Buffers */
    EC_T_DWORD                  dwTxBuffers;            /* [in]     amount of Transmit Buffers (Frames) */

    PF_emllISRCallback          pfIsrCallback;          /* [in]     optional called by ISR */
    EC_T_PVOID                  pvIsrCBContext;         /* [in]     ISR callback context */
} EC_PACKED EC_T_LINK_OPENPARMS_RTL8139;


/* NDISUIO Windows CE   */
/* ==================== */

#define EC_LINK_TYPE_NDISUIO    0xA1ECA101
#define ECAT_NDISUIO_DEVNAME    TEXT("ECT1:")

/* Link parameters */
typedef struct _EC_T_LINK_OPENPARMS_NDISUIO
{
    EC_T_DWORD      dwType;
    EC_T_DWORD      dwRcvThreadPriority;
    EC_T_WCHAR*     szNetworkAdapterName;
} EC_PACKED EC_T_LINK_OPENPARMS_NDISUIO;


/* Davicom DM9000       */
/* ==================== */

#define EC_LINK_TYPE_DM9000    0xA1ECA104

/* Link parameters */
typedef struct _EC_T_LINK_OPENPARMS_DM9000
{
    EC_T_DWORD      dwType;
    EC_T_DWORD      dwRcvThreadPriority;
    EC_T_DWORD      dwBaseAddress;
    EC_T_DWORD      dwBaseIO;
    EC_T_LINKMODE   eLinkMode;
    EC_T_DWORD      dwIrqNum;
    EC_T_DWORD      dwSysIntr;

    EC_T_DWORD      dwFrameBufferLen;

    PF_DOINT_HDL	pfDoIntHandling;	/* [out]	interrupt processing routine for ext IntHandling */
} EC_PACKED EC_T_LINK_OPENPARMS_DM9000;

/* Realtek Gigabit family */
/* ====================== */

#define EC_LINK_TYPE_RTLGIG     0xA1ECA10C

/* Link parameters */
typedef struct _EC_T_LINK_OPENPARMS_RTLGIG
{
    EC_T_DWORD                  dwType;         /* 0xA1ECA10C */
    EC_T_DWORD                  dwUnit;
    
    EC_T_DWORD                  dwIstPriority;
    EC_T_LINKMODE               eLinkMode;
    PF_DOINT_HDL                pfDoIntHandling;        /* [out]    interrupt processing routine for ext IntHandling */
    
    EC_T_DWORD                  dwRxBuffers;            /* [in]     amount of Receive  Buffers */
    EC_T_DWORD                  dwTxBuffers;            /* [in]     amount of Transmit Buffers (Frames) */
    
    PF_emllISRCallback          pfIsrCallback;          /* [in]     optional called by ISR */
    EC_T_PVOID                  pvIsrCBContext;         /* [in]     ISR callback context */
    EC_T_BOOL                   bDisableLocks;          /* [in]     Locks in LL Disabled */
} EC_PACKED EC_T_LINK_OPENPARMS_RTLGIG;

#define EC_LINK_TYPE_EOE    0xA1ECA201

/* Link parameters */
/*****************************************************************************/
/** \brief  Variables to identify the EOE link layer driver/instance which shall
*           be opened with the ``EoELinkOpen()`` call.
*
*           Attention: Parameters not used for identification of the link layer
*                      (or master stack  instance) must be cleared to 0.
*/
typedef struct _EC_T_LINK_OPENPARMS_EOE
{
    /*! \var    dwEmInstanceID
        \brief  Instance ID (or instance number), identical to the instance ID
                of the EtherCAT master which is the master of the network we want
                to open.
	*/
    EC_T_DWORD dwEmInstanceID;

    /*! \var    pvUplinkInstance
        \brief  Pointer to the ``CEcEoEUplink`` instance/object (if available).
	*/
    EC_T_PVOID pvUplinkInstance;

    /*! \var    dwEthAddress
        \brief  Ethernet address of the driver/adapter we want to open.
	*/
    EC_T_BYTE abyEthAddress[6];

    /*! \var    abyIpAdress
        \brief  IP address of the driver/adapter we want to open.
	*/
    EC_T_BYTE abyIpAdress[4];

    /*! \var    szEoEDrvIdent
        \brief  Name of the driver/adapter we want to open (e.g. "eth0").
	*/
    EC_T_CHAR szEoEDrvIdent[MAX_DRIVER_IDENT_LEN];
} EC_PACKED EC_T_LINK_OPENPARMS_EOE;







/*****************************************************************************/
/** \brief  Structure to open the link layer. This struct contains of two
*           parts. A common part that is equal for all Link Layer and an
*           driver specific part. 
*/

#define MAX_SIZE_DRV_SPECIFIC_PARM  ((EC_T_INT)256)
#define MAX_SIZE_LINK_OPEN_PARM     (sizeof(EC_T_LINK_DEV_OPEN_PARAM))

typedef struct _EC_T_LINK_DEV_OPEN_PARAM
{
    EC_T_LINK_DEV_PARAM oLinkDevParam;

    union _EC_T_LINK_OPEN_PARMS
    {
        EC_T_LINK_OPENPARMS_I8255x   oI8255x;
        EC_T_LINK_OPENPARMS_I8254x   oI8254x;
        EC_T_LINK_OPENPARMS_RTL8139  oRtl8139;
        EC_T_LINK_OPENPARMS_NDISUIO  oNdisUio;
        EC_T_LINK_OPENPARMS_DM9000   oDm9000;
        EC_T_LINK_OPENPARMS_WINPCAP  oWinPcap;
        EC_T_LINK_OPENPARMS_SNARF    oSnarf;
        EC_T_LINK_OPENPARMS_ETHERLIB oEtherlib;
        EC_T_LINK_OPENPARMS_EOE      oEoE;
        EC_T_BYTE                    abyOpenParam[MAX_SIZE_DRV_SPECIFIC_PARM];
    } linkOpenParms;
} EC_PACKED EC_T_LINK_DEV_OPEN_PARAM;



/*-FUNCTIONS DECLARATION-----------------------------------------------------*/

ATEMLL_API EC_T_DWORD emllRegisterI8255x
    (EC_T_LINK_DRV_DESC*  pLinkDrvDesc        /* [in,out] link layer driver descriptor */
    ,EC_T_DWORD         dwLinkDrvDescSize);   /* [in]     size in bytes of link layer driver descriptor */

ATEMLL_API EC_T_DWORD emllRegisterI8254x
    (EC_T_LINK_DRV_DESC*  pLinkDrvDesc        /* [in,out] link layer driver descriptor */
    ,EC_T_DWORD         dwLinkDrvDescSize);   /* [in]     size in bytes of link layer driver descriptor */

ATEMLL_API EC_T_DWORD emllRegisterRTL8139
    (EC_T_LINK_DRV_DESC*  pLinkDrvDesc        /* [in,out] link layer driver descriptor */
    ,EC_T_DWORD         dwLinkDrvDescSize);   /* [in]     size in bytes of link layer driver descriptor */

ATEMLL_API EC_T_DWORD emllRegisterNdisUio
    (EC_T_LINK_DRV_DESC*  pLinkDrvDesc        /* [in,out] link layer driver descriptor */
    ,EC_T_DWORD         dwLinkDrvDescSize);   /* [in]     size in bytes of link layer driver descriptor */

#ifdef __cplusplus
}
#endif


/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#undef  EC_PACKED

#if (defined __RCX__)
#pragma pack ()
#endif

#elif defined _MSC_VER  /* MICROSOFT */

#undef  EC_PACKED
#pragma pack (pop)

#elif defined __MET__

#undef EC_PACKED
/*#pragma pack (pop)*/
#pragma Pop_align_members()

#else
#endif

#endif /* INC_ECLINK */


/*-END OF SOURCE FILE--------------------------------------------------------*/
